package gihyo.lucene.ch3;

/**
 * Copyright 2006 SEKIGUCHI, Koji
 * 
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *  http://www.apache.org/licenses/LICENSE-2.0
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

import java.io.IOException;
import java.util.Date;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.store.Directory;

public abstract class AbstractIndexer {

    protected static final String DEFAULT_ANALYZER = "org.apache.lucene.analysis.ja.JapaneseAnalyzer";
    protected static final String DEFAULT_FIELD_STORE = "YES";
    protected static final String FIELD_STORE_COMPRESS = "COMPRESS";

    protected static final String PROP_ANALYZER = "analyzer";
    protected static final String PROP_MERGE_FACTOR = "merge.factor";
    protected static final String PROP_MAX_MERGE_DOCS = "max.merge.docs";
    protected static final String PROP_MAX_BUFFERED_DOCS = "max.buffered.docs";
    protected static final String PROP_FIELD_STORE = "field.store";

    protected Analyzer analyzer;
    protected Field.Store fieldStore;

    protected Analyzer getAnalyzer() throws IndexerException {
	try{
	    if( analyzer == null ){
		String ana = System.getProperty( PROP_ANALYZER, DEFAULT_ANALYZER );
		analyzer = (Analyzer)getClass().getClassLoader().loadClass( ana ).newInstance();
		System.out.println( "* Analyzer : " + ana );
	    }
	    return analyzer;
	}
	catch( ClassNotFoundException e ){
	    throw new IndexerException( e );
	}
	catch( InstantiationException e ){
	    throw new IndexerException( e );
	}
	catch( IllegalAccessException e ){
	    throw new IndexerException( e );
	}
    }

    protected void makeIndex() throws IndexerException {

	long s = System.currentTimeMillis();
	Date start = new Date( s );
	System.out.println( "* Jn : " + start );

	IndexWriter writer = getIndexWriter();
	begin();
	try{
	    while( hasNext() ){
		Object record = next();
		writer.addDocument( getDocument( record ) );
	    }
	    writer.optimize();
	}
	catch( IOException e ){
	    throw new IndexerException( e );
	}
	finally{
	    try{
		writer.close();
	    }
	    catch( IOException ignore ){}
	    end();
	}

	long e = System.currentTimeMillis();
	Date end = new Date( e );
	long elapse = e - s;
	System.out.println( "* I : " + end + "(oߎ : " + elapse + "~b)" );
    }

    protected IndexWriter getIndexWriter() throws IndexerException {
	IndexWriter writer = null;
	try{
	    writer = new IndexWriter( getDirectory(), getAnalyzer(), true );
	}
	catch( IOException e ){
	    throw new IndexerException( e );
	}
	String mergeFactor = System.getProperty( PROP_MERGE_FACTOR );
	if( mergeFactor != null )
	    writer.setMergeFactor( Integer.parseInt( mergeFactor ) );
	String maxMergeDocs = System.getProperty( PROP_MAX_MERGE_DOCS );
	if( maxMergeDocs != null )
	    writer.setMaxMergeDocs( Integer.parseInt( maxMergeDocs ) );
	String maxBufferedDocs = System.getProperty( PROP_MAX_BUFFERED_DOCS );
	if( maxBufferedDocs != null )
	    writer.setMaxBufferedDocs( Integer.parseInt( maxBufferedDocs ) );
	System.out.println( "* }[WW : " + Integer.toString( writer.getMergeFactor() ) );
	System.out.println( "* ő}[WhLg : " + Integer.toString( writer.getMaxMergeDocs() ) );
	System.out.println( "* őobt@hLg : " + Integer.toString( writer.getMaxBufferedDocs() ) );
	return writer;
    }

    protected Field.Store getFieldStore(){
	if( fieldStore == null ){
	    fieldStore = Field.Store.YES;
	    String fs = System.getProperty( PROP_FIELD_STORE, DEFAULT_FIELD_STORE );
	    System.out.print( "* k[h : " );
	    if( FIELD_STORE_COMPRESS.equalsIgnoreCase( fs ) ){
		System.out.println( "YES" );
		fieldStore = Field.Store.COMPRESS;
	    }
	    else
		System.out.println( "NO" );
	}
	return fieldStore;
    }

    protected void begin() throws IndexerException {}
    protected void end() throws IndexerException {}

    protected abstract boolean hasNext() throws IndexerException;
    protected abstract Object next() throws IndexerException;
    protected abstract Directory getDirectory() throws IndexerException;
    protected abstract Document getDocument( final Object record ) throws IndexerException;
}
