package gihyo.lucene.ch4;

/**
 * Copyright 2006 SEKIGUCHI, Koji
 * 
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *  http://www.apache.org/licenses/LICENSE-2.0
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

import java.io.IOException;
import java.text.DecimalFormat;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Hits;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.Explanation;
import org.apache.lucene.document.Document;
import org.apache.lucene.store.Directory;

public abstract class AbstractSearcher {

    protected static final String DEFAULT_ANALYZER = "org.apache.lucene.analysis.ja.JapaneseAnalyzer";

    protected static final String PROP_ANALYZER = "analyzer";
    protected static final String PROP_SUMMARY = "summary";
    protected static final String PROP_EXPLANATION = "explanation";

    protected Analyzer analyzer;
    protected boolean summary = Boolean.getBoolean( PROP_SUMMARY );
    protected boolean explanation = Boolean.getBoolean( PROP_EXPLANATION );
    protected DecimalFormat formatter = new DecimalFormat( "0.0########" );

    protected Analyzer getAnalyzer() throws SearcherException {
	try{
	    if( analyzer == null ){
		String ana = System.getProperty( PROP_ANALYZER, DEFAULT_ANALYZER );
		analyzer = (Analyzer)getClass().getClassLoader().loadClass( ana ).newInstance();
		System.out.println( "* Analyzer : " + ana );
	    }
	    return analyzer;
	}
	catch( ClassNotFoundException e ){
	    throw new SearcherException( e );
	}
	catch( InstantiationException e ){
	    throw new SearcherException( e );
	}
	catch( IllegalAccessException e ){
	    throw new SearcherException( e );
	}
    }

    protected void searchIndex( final Query query ) throws SearcherException {

	System.out.println( "Query = \"" + query.toString() + "\"" );
	IndexSearcher searcher = null;
	try{
	    searcher = new IndexSearcher( getDirectory() );
	    Hits hits = searcher.search( query );
	    printHits( searcher, query, hits );
	}
	catch( IOException e ){
	    throw new SearcherException( e );
	}
	finally{
	    try{
		if( searcher != null )
		    searcher.close();
	    }
	    catch( IOException ignore ){}
	}
    }

    protected void printHits( final IndexSearcher searcher, final Query query, final Hits hits )
	throws SearcherException {
	int length = hits.length();
	System.out.println( Integer.toString( length ) + " qbg܂B" );
	try{
	    for( int i = 0; i < length; i++ ){
		Document doc = hits.doc( i );
		if( summary )
		    printSummary( doc );
		else
		    printDocument( doc );
		int id = hits.id( i );
		System.out.println( "ID = " + Integer.toString( id ) + ", XRA = " + formatter.format( hits.score( i ) ) );
		if( explanation ){
		    Explanation e = searcher.explain( query, id );
		    System.out.println( e.toString() );
		}
	    }
	}
	catch( IOException e ){
	    throw new SearcherException( e );
	}
    }

    protected abstract void printDocument( final Document doc );
    protected abstract void printSummary( final Document doc );
    protected abstract Directory getDirectory() throws SearcherException;
}
