package gihyo.lucene.ch6;

/**
 * Copyright 2006 SEKIGUCHI, Koji
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.IOException;
import java.util.Stack;
import org.apache.lucene.analysis.TokenStream;
import org.apache.lucene.analysis.TokenFilter;
import org.apache.lucene.analysis.Token;

public class ThesaurusFilter extends TokenFilter {
  public static final String TOKEN_TYPE_THESAURUS = "THESAURUS";

  private Stack thesaurusStack;
  private Thesaurus thesaurus;

  public ThesaurusFilter(TokenStream in, Thesaurus thesaurus) {
    super(in);
    thesaurusStack = new Stack();
    this.thesaurus = thesaurus;
  }

  public Token next() throws IOException {
    if (thesaurusStack.size() > 0) {
      return (Token) thesaurusStack.pop();
    }

    Token token = input.next();
    if (token == null) {
      return null;
    }

    addAliasesToStack(token);

    return token;
  }

  private void addAliasesToStack(Token token) throws IOException {
    String[] words = thesaurus.getGroupWords(token.termText());

    if (words == null) return;

    for (int i = 0; i < words.length; i++) {
      Token synToken = new Token(words[i],
                                 token.startOffset(),
                                 token.endOffset(),
                                 TOKEN_TYPE_THESAURUS);
      synToken.setPositionIncrement(0);

      thesaurusStack.push(synToken);
    }
  }
}
