package filters;

import java.io.IOException;
import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.ServletException;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.hibernate.HibernateException;
import org.hibernate.cfg.Configuration;

public class OpenSessionInView implements Filter {

    /**
     * Holds the current hibernate session, if one has been created.
     */
    protected static ThreadLocal hibernateHolder = new ThreadLocal(); 
    
    /**
     */
    protected static SessionFactory sessionFactory = null;

    public void init( FilterConfig filterConfig ) throws ServletException {
	try{
	    if( sessionFactory == null ){
		Configuration cfg = new Configuration();
		cfg.configure();
		sessionFactory = cfg.buildSessionFactory();
	    }
	}
	catch( HibernateException e ){
	    throw new ServletException( e );
	}
    }

    /**
     */
    public void doFilter( ServletRequest request,
			  ServletResponse response,
			  FilterChain chain )
	throws IOException, ServletException {
        if( hibernateHolder.get() != null )
            throw new IllegalStateException(
                "A session is already associated with this thread!  "
                + "Someone must have called getSession() outside of the context "
                + "of a servlet request.");
        try{
            chain.doFilter( request, response );
        }
        finally{
            Session sess = (Session)hibernateHolder.get();
            if( sess != null ){
                hibernateHolder.set( null );
                try{
                    sess.close();
                }
                catch( HibernateException ex ){
		    throw new ServletException( ex );
		}
            }
        }
    }

    /**
     * ONLY ever call this method from within the context of a servlet request
     * (specifically, one that has been associated with this filter).  If you
     * want a Hibernate session at some other time, call getSessionFactory()
     * and open/close the session yourself.
     *
     * @return an appropriate Session object
     */
    public static Session getSession() throws HibernateException {
        Session sess = (Session)hibernateHolder.get();
        if( sess == null ){
            sess = sessionFactory.openSession();
            hibernateHolder.set( sess );
        }
        return sess;
    }
    
    /**
     * @return the hibernate session factory
     */
    public static SessionFactory getSessionFactory(){
        return sessionFactory;
    }
    
    /**
     * This is a simple method to reduce the amount of code that needs
     * to be written every time hibernate is used.
     */
    public static void rollback( Transaction tx ){
        if( tx != null ){
            try{
                tx.rollback();
            }
            catch( HibernateException ex ){
                // Probably don't need to do anything - this is likely being
                // called because of another exception, and we don't want to
                // mask it with yet another exception.
            }
        }
    }

    /**
     */
    public void destroy(){
        // Nothing necessary
    }
}
