package nayami.services;

/**
 * Copyright 2006 SEKIGUCHI, Koji
 * 
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *  http://www.apache.org/licenses/LICENSE-2.0
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

import java.text.DateFormat;
import java.io.IOException;
import java.util.List;
import java.util.ArrayList;
import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.DateTools;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.Hits;
import org.apache.lucene.index.Term;
import org.apache.lucene.queryParser.QueryParser;
import org.apache.lucene.queryParser.ParseException;
import nayami.constant.DocumentTypes;
import nayami.constant.DocumentFields;

public class SearchServices {

    private static Log log = LogFactory.getLog( SearchServices.class );
    private static DateFormat formatter = DateFormat.getDateInstance();

    private ConfigServices cs;
    private int pageSize;
    private SearcherFactory searcherFactory;
    private Analyzer analyzer;

    private SearchServices( HttpServletRequest request ) throws SystemException {
	cs = ConfigServices.getInstance( request );
	ServletContext sc = request.getSession().getServletContext();
	pageSize = Integer.parseInt( (String)sc.getAttribute( "pageSize" ) );
	searcherFactory = (SearcherFactory)sc.getAttribute( SearcherFactory.SEARCHER_FACTORY );
	if( searcherFactory == null )
	    throw new SystemException( "SearcherFactorynullłBLucenȅɎsĂ܂B" );
	analyzer = (Analyzer)sc.getAttribute( "analyzer" );
	if( analyzer == null )
	    throw new SystemException( "AnalyzernullłBLucenȅɎsĂ܂B" );
    }

    public static SearchServices getInstance( HttpServletRequest request ) throws SystemException {
	return new SearchServices( request );
    }

    public SearchResult search( String q, boolean flagArticle, boolean flagFaq, boolean flagBook, boolean flagBulletin, int begin ) throws SystemException, QueryParserException {
	QueryParser qp = new QueryParser( DocumentFields.CONTENT, analyzer );
	Query source = null;
	try{
	    source = qp.parse( q );
	}
	catch( ParseException e ){
	    throw new QueryParserException( e );
	}
	IndexSearcher searcher = null;

	Query query = source;
	if( !flagArticle || !flagFaq || !flagBook || !flagBulletin ){
	    log.debug( "BooleanQuery쐬܂B" );
	    query = new BooleanQuery();
	    ((BooleanQuery)query).add( source, BooleanClause.Occur.MUST );
	    if( !flagArticle )
		((BooleanQuery)query).add( new TermQuery( new Term( DocumentFields.TYPE, DocumentTypes.ARTICLE ) ), BooleanClause.Occur.MUST_NOT );
	    if( !flagFaq )
		((BooleanQuery)query).add( new TermQuery( new Term( DocumentFields.TYPE, DocumentTypes.FAQ ) ), BooleanClause.Occur.MUST_NOT );
	    if( !flagBook )
		((BooleanQuery)query).add( new TermQuery( new Term( DocumentFields.TYPE, DocumentTypes.BOOK ) ), BooleanClause.Occur.MUST_NOT );
	    if( !flagBulletin )
		((BooleanQuery)query).add( new TermQuery( new Term( DocumentFields.TYPE, DocumentTypes.BULLETIN ) ), BooleanClause.Occur.MUST_NOT );
	}
	log.info( "query : " + query );

	try{
	    searcher = searcherFactory.createSearcher();
	    Hits hits = searcher.search( query );
	    List items = new ArrayList();
	    int length = hits.length();
	    for( int i = begin; i < Math.min( begin + pageSize, length ); i++ ){
		Document doc = hits.doc( i );
		ItemBean item = new ItemBean();
		item.setUrl( doc.get( DocumentFields.URL ) );
		item.setTitle( doc.get( DocumentFields.TITLE ) );
		item.setDate( formatter.format( DateTools.stringToDate( doc.get( DocumentFields.DATE ) ) ) );
		item.setScore( hits.score( i ) );
		items.add( item );
	    }
	    return new SearchResult( length, items );
	}
	catch( java.text.ParseException e ){
	    throw new SystemException( e );
	}
	catch( IOException e ){
	    throw new SystemException( e );
	}
	finally{
	    if( searcher != null ){
		try{
		    searcher.close();
		}
		catch( IOException e ){
		    throw new SystemException( e );
		}
	    }
	}
    }
}
